<?php
// Common functions for the application

function get_setting($key) {
    $db = get_db_connection();
    $stmt = $db->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
    $stmt->bind_param('s', $key);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        return $row['setting_value'];
    }
    return null;
}

function update_setting($key, $value) {
    $db = get_db_connection();
    $stmt = $db->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
    $stmt->bind_param('ss', $key, $value);
    return $stmt->execute();
}

function is_logged_in() {
    return isset($_SESSION['admin_id']);
}

function auth_guard() {
    if (!is_logged_in()) {
        header('Location: ' . APP_URL . '/admin/login');
        exit;
    }
}

function view($view_name, $data = []) {
    extract($data);
    include_once __ROOT__ . '/admin/views/' . $view_name . '.php';
}

// --- Payment Gateway Functions ---
function get_available_gateways() {
    $gateways = [];
    $plugin_dir = __ROOT__ . '/content/plugins/gateways/';
    if(!is_dir($plugin_dir)) return [];
    $folders = array_diff(scandir($plugin_dir), array('..', '.'));

    foreach ($folders as $folder) {
        if (is_dir($plugin_dir . $folder)) {
            $info_file = $plugin_dir . $folder . '/info.json';
            if (file_exists($info_file)) {
                $info = json_decode(file_get_contents($info_file), true);
                if ($info) {
                    $gateways[$info['slug']] = $info;
                }
            }
        }
    }
    return $gateways;
}

function get_gateway_info($slug) {
    $gateways = get_available_gateways();
    return $gateways[$slug] ?? null;
}

function get_gateway_settings($slug) {
    $db = get_db_connection();
    $key = 'gateway_' . $slug;
    $stmt = $db->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
    $stmt->bind_param('s', $key);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        return json_decode($row['setting_value'], true);
    }
    return [];
}

function update_gateway_settings($slug, $settings) {
    $db = get_db_connection();
    $key = 'gateway_' . $slug;
    $value = json_encode($settings);
    $stmt = $db->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
    $stmt->bind_param('ss', $key, $value);
    return $stmt->execute();
}
